/*
 * Decompiled with CFR 0.152.
 */
package org.conscrypt;

import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Objects;
import javax.crypto.BadPaddingException;
import org.conscrypt.HpkeDecryptException;
import org.conscrypt.HpkeSpi;
import org.conscrypt.HpkeSuite;
import org.conscrypt.NativeCrypto;
import org.conscrypt.NativeRef;
import org.conscrypt.OpenSSLX25519PrivateKey;
import org.conscrypt.OpenSSLX25519PublicKey;
import org.conscrypt.Preconditions;

public class HpkeImpl
implements HpkeSpi {
    private final HpkeSuite hpkeSuite;
    private NativeRef.EVP_HPKE_CTX ctx;
    private byte[] encapsulated = null;

    public HpkeImpl(HpkeSuite hpkeSuite) {
        this.hpkeSuite = hpkeSuite;
    }

    @Override
    public void engineInitSender(PublicKey recipientKey, byte[] info, PrivateKey senderKey, byte[] psk, byte[] psk_id) throws InvalidKeyException {
        this.checkNotInitialised();
        this.checkArgumentsForBaseModeOnly(senderKey, psk, psk_id);
        if (recipientKey == null) {
            throw new InvalidKeyException("null recipient key");
        }
        if (!(recipientKey instanceof OpenSSLX25519PublicKey)) {
            throw new InvalidKeyException("Unsupported recipient key class: " + recipientKey.getClass());
        }
        byte[] recipientKeyBytes = ((OpenSSLX25519PublicKey)recipientKey).getU();
        Object[] result = NativeCrypto.EVP_HPKE_CTX_setup_base_mode_sender(this.hpkeSuite, recipientKeyBytes, info);
        this.ctx = (NativeRef.EVP_HPKE_CTX)result[0];
        this.encapsulated = (byte[])result[1];
    }

    @Override
    public void engineInitSenderForTesting(PublicKey recipientKey, byte[] info, PrivateKey senderKey, byte[] psk, byte[] psk_id, byte[] sKe) throws InvalidKeyException {
        this.checkNotInitialised();
        Objects.requireNonNull(sKe);
        this.checkArgumentsForBaseModeOnly(senderKey, psk, psk_id);
        if (recipientKey == null) {
            throw new InvalidKeyException("null recipient key");
        }
        if (!(recipientKey instanceof OpenSSLX25519PublicKey)) {
            throw new InvalidKeyException("Unsupported recipient key class: " + recipientKey.getClass());
        }
        byte[] recipientKeyBytes = ((OpenSSLX25519PublicKey)recipientKey).getU();
        Object[] result = NativeCrypto.EVP_HPKE_CTX_setup_base_mode_sender_with_seed_for_testing(this.hpkeSuite, recipientKeyBytes, info, sKe);
        this.ctx = (NativeRef.EVP_HPKE_CTX)result[0];
        this.encapsulated = (byte[])result[1];
    }

    @Override
    public void engineInitRecipient(byte[] encapsulated, PrivateKey recipientKey, byte[] info, PublicKey senderKey, byte[] psk, byte[] psk_id) throws InvalidKeyException {
        this.checkNotInitialised();
        this.checkArgumentsForBaseModeOnly(senderKey, psk, psk_id);
        Preconditions.checkNotNull(encapsulated, "null encapsulated data");
        if (encapsulated.length != this.hpkeSuite.getKem().getEncapsulatedLength()) {
            throw new InvalidKeyException("Invalid encapsulated length: " + encapsulated.length);
        }
        if (recipientKey == null) {
            throw new InvalidKeyException("null recipient key");
        }
        if (!(recipientKey instanceof OpenSSLX25519PrivateKey)) {
            throw new InvalidKeyException("Unsupported recipient key class: " + recipientKey.getClass());
        }
        byte[] recipientKeyBytes = ((OpenSSLX25519PrivateKey)recipientKey).getU();
        this.ctx = (NativeRef.EVP_HPKE_CTX)NativeCrypto.EVP_HPKE_CTX_setup_base_mode_recipient(this.hpkeSuite, recipientKeyBytes, encapsulated, info);
    }

    private void checkArgumentsForBaseModeOnly(Key senderKey, byte[] psk, byte[] psk_id) {
        if (senderKey != null) {
            throw new UnsupportedOperationException("Asymmetric authentication not supported");
        }
        Objects.requireNonNull(psk);
        Objects.requireNonNull(psk_id);
        if (psk.length > 0 || psk_id.length > 0) {
            throw new UnsupportedOperationException("PSK authentication not supported");
        }
    }

    @Override
    public byte[] engineSeal(byte[] plaintext, byte[] aad) {
        this.checkIsSender();
        Preconditions.checkNotNull(plaintext, "null plaintext");
        return NativeCrypto.EVP_HPKE_CTX_seal(this.ctx, plaintext, aad);
    }

    @Override
    public byte[] engineExport(int length, byte[] exporterContext) {
        this.checkInitialised();
        long maxLength = this.hpkeSuite.getKdf().maxExportLength();
        if (length < 0 || (long)length > maxLength) {
            throw new IllegalArgumentException("Export length must be between 0 and " + maxLength + ", but was " + length);
        }
        return NativeCrypto.EVP_HPKE_CTX_export(this.ctx, exporterContext, length);
    }

    @Override
    public byte[] engineOpen(byte[] ciphertext, byte[] aad) throws GeneralSecurityException {
        this.checkIsRecipient();
        Preconditions.checkNotNull(ciphertext, "null ciphertext");
        try {
            return NativeCrypto.EVP_HPKE_CTX_open(this.ctx, ciphertext, aad);
        }
        catch (BadPaddingException e) {
            throw new HpkeDecryptException(e.getMessage());
        }
    }

    private void checkInitialised() {
        if (this.ctx == null) {
            throw new IllegalStateException("Not initialised");
        }
    }

    private void checkNotInitialised() {
        if (this.ctx != null) {
            throw new IllegalStateException("Already initialised");
        }
    }

    private void checkIsSender() {
        this.checkInitialised();
        if (this.encapsulated == null) {
            throw new IllegalStateException("Internal error");
        }
    }

    private void checkIsRecipient() {
        this.checkInitialised();
        if (this.encapsulated != null) {
            throw new IllegalStateException("Internal error");
        }
    }

    @Override
    public byte[] getEncapsulated() {
        this.checkIsSender();
        return this.encapsulated;
    }

    public static class X25519_CHACHA20
    extends HpkeImpl {
        public X25519_CHACHA20() {
            super(new HpkeSuite(32, 1, 3));
        }
    }

    public static class X25519_AES_256
    extends HpkeImpl {
        public X25519_AES_256() {
            super(new HpkeSuite(32, 1, 2));
        }
    }

    public static class X25519_AES_128
    extends HpkeImpl {
        public X25519_AES_128() {
            super(new HpkeSuite(32, 1, 1));
        }
    }
}

